
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <mpi.h> 
#include <sys/time.h>

double eps = 0.001;
double delta_t = 0.000001;
double alpha = 1;

typedef struct {
	int start_m;	// first tile to process in X-axis
	int start_n;	// first tile to process in Y-axis
	int end_m;	// last tile to process in X-axis
	int end_n;	// last tile to process in Y-axis
} t_process_info;

int main(int argc, char **argv)
{
	MPI_Init(&argc, &argv);
	int i, j, m, n, err;
	double **a;
	double start, end;
	t_process_info pi;

	if ((argc < 3) || (argc > 5)) {
		fprintf(stderr, "Usage: heat <m> <n> [<epsilon> [delta_t]]!\n\n");
		fprintf(stderr, "   <m>       -- X-Size of matrix\n");
		fprintf(stderr, "   <n>       -- Y-Size of matrix\n");
		fprintf(stderr, "   <epsilon> -- accuracy parameter\n");
		fprintf(stderr, "   <delta_t> -- time step\n");
		exit(1);
	}

	/*
	** Erstes und zweites Argument: Größe der Matrix
	*/
	m = atoi(argv[1]);
	if ((m < 3) || (m > 6000)) {
		fprintf(stderr, "Error: size m out of range [3 .. 6000] !\n");
		exit(1);
	}
	n = atoi(argv[2]);
	if ((n < 3) || (n > 6000)) {
		fprintf(stderr, "Error: size n out of range [3 .. 6000] !\n");
		exit(1);
	}
	/*
	** Drittes (optionales) Argument: "Genauigkeitsfaktor" eps
	*/
	if (argc >= 4) {
		eps = atof(argv[3]);
	}
	if ((eps <= 0) || (eps > 0.01)){
		fprintf(stderr, "Error: epsilon must be between 0.0000000001 and 0.01 )\n");
		exit(1);
	}
	/*
	** Viertes (optionales) Argument: delta_t
	*/
	if (argc >= 5) {
		delta_t = atof(argv[4]);
	}
	if ((delta_t <= 0) || (delta_t > 1.0)){
		fprintf(stderr, "Error: delta_t (%f) must be between 0.0000000001 and 1.0\n", delta_t);
		exit(1);
	}

	int pid, num_p;
	if((err = MPI_Comm_rank(MPI_COMM_WORLD, &pid))) {
		fprintf(stderr, "Cannot fetch PID due to %d\n", err);
		exit(1);
	}

	if(MPI_Comm_size(MPI_COMM_WORLD, &num_p)) {
		fprintf(stderr, "Cannot fetch size of cluster\n");		
		exit(1);
	}

	if(pid == 0) {
		printf("number of processes: %d\n", num_p);
	}

	int pro_per_dim[2];
	float temp_f,temp_g;
	temp_g = sqrt((float)(num_p*m)/n);
	temp_f = num_p/temp_g;
	temp_g = floor(temp_g);
	temp_f = floor(temp_f);
	pro_per_dim[0] = (int)temp_f;
	pro_per_dim[1] = (int)temp_g;

	int m_per_pro, n_per_pro;
	m_per_pro = ceil(m/(float)pro_per_dim[1]);
	n_per_pro = ceil(n/(float)pro_per_dim[0]);

	if(pid==0) {
		printf("dim0: %d dim1: %d\n", pro_per_dim[0], pro_per_dim[1]);
		printf("size per pro: %dx%d\n", m_per_pro, n_per_pro);	
	}
	
	int periods[] = {0,0};	// edges are not connected
	MPI_Comm cart_comm;
	if(MPI_Cart_create(MPI_COMM_WORLD, 2, pro_per_dim, periods, 0, &cart_comm)) {
		fprintf(stderr, "Cannot create topology\n");		
		exit(1);
	}
	if(cart_comm == MPI_COMM_NULL) {
		printf("process %d not in use. exiting...\n", pid);
		MPI_Finalize();
		exit(0);
	}
	int coord[2];
	if(MPI_Cart_coords(cart_comm, pid, 2, coord)) {
		fprintf(stderr, "Cannot get coordinates\n");		
		exit(1);
	}

	// calculate own field using coord
	pi.start_m = coord[0] * m_per_pro;
	pi.end_m = (coord[0]+1)*m_per_pro -1;
	pi.start_n = coord[1] * n_per_pro;
	pi.end_n = (coord[0]+1) * n_per_pro -1;

	// print own info
	printf("pid(%d,%d) from (%d, %d) to (%d,%d)\n",
		coord[0],
		coord[1],
		pi.start_n,
		pi.start_m,
		pi.end_n,
		pi.end_m
	);

	// write neighbar_com function

	MPI_Finalize();

	return 0;
}

