#define min(X,Y) (((X) < (Y)) ? (X) : (Y))
#define max(X,Y) (((X) > (Y)) ? (X) : (Y))

#include <omp.h>
#include <mpi.h> 
#include <stdio.h>
#include <stdlib.h>
#include "project.h"

void Alloc_Partial_Field(int *matrix_size, double ***partial_field, double ***partial_field_clipboard) {
	*partial_field = New_Matrix(matrix_size[0], matrix_size[1]);
	if (partial_field == NULL) {
		fprintf(stderr, "Can't allocate partial_field\n");
		exit(1);
	}
	*partial_field_clipboard = New_Matrix(matrix_size[0], matrix_size[1]);
	if (partial_field_clipboard == NULL) {
		fprintf(stderr, "Can't allocate partial_field_clipboard\n");
		exit(1);
	}

	Init_Matrix(*partial_field, matrix_size[0], matrix_size[1], 0);
	Init_Matrix(*partial_field_clipboard, matrix_size[0], matrix_size[1], 0);
}

void Init_Jacobi(int dim0_size, int dim1_size, int alpha, double *delta_t, double *hx, double *hy, double *hx_square, double *hy_square) {
	*hx = 1.0/(double)dim0_size;
	*hy = 1.0/(double)dim1_size;
	*hx_square = *hx * *hx;
	*hy_square = *hy * *hy;

	double max_delta_t = 0.25*((min(*hx,*hy))*(min(*hx,*hy)))/alpha;  /* minimaler Wert für Konvergenz */
	if (*delta_t > max_delta_t) { 
		*delta_t = max_delta_t;
		printf ("Info: delta_t set to %.10lf.\n", *delta_t);
	}
}

void Init_Edges(int dim0_size, int dim1_size, int *matrix_size, int neighbor_dim0_left, int neighbor_dim0_right, int neighbor_dim1_left, int neighbor_dim1_right, double **partial_field, double **partial_field_clipboard, t_process_info pi) {
	// determine wether edge is edge of root field
	if(neighbor_dim1_left == MPI_PROC_NULL) {
		for(int i = pi.start_m; i <= pi.end_m; i++) {
			partial_field[i - pi.start_m + 1][1] = (double)i / (dim0_size-1);
			partial_field_clipboard[i - pi.start_m + 1][1] = (double)i / (dim0_size-1);
		}
	}
	if(neighbor_dim1_right == MPI_PROC_NULL) {
		for(int i = pi.start_m; i <= pi.end_m; i++) {
			partial_field[i - pi.start_m + 1][matrix_size[1]-2] = 1 - (double)i / (dim0_size-1);
			partial_field_clipboard[i - pi.start_m + 1][matrix_size[1]-2] = 1 - (double)i / (dim0_size-1);
		}
	}
	if(neighbor_dim0_left == MPI_PROC_NULL) {
		for(int i = pi.start_n; i <= pi.end_n; i++) {
			partial_field[1][i - pi.start_n + 1] = (double)i / (dim1_size-1);
			partial_field_clipboard[1][i - pi.start_n + 1] = (double)i / (dim1_size-1);
		}
	}
	if(neighbor_dim0_right == MPI_PROC_NULL) {
		for(int i = pi.start_n; i <= pi.end_n; i++) {
			partial_field[matrix_size[0] - 2][i - pi.start_n + 1] = 1 - (double)i / (dim1_size-1);
			partial_field_clipboard[matrix_size[0] - 2][i - pi.start_n + 1] = 1 - (double)i / (dim1_size-1);
		}
	}
}

int Jacobi_Iterate(int neighbor_dim0_left, int neighbor_dim0_right, int neighbor_dim1_left, int neighbor_dim1_right, double alpha, double delta_t, double eps, double hx_square, double hy_square, t_process_info pi, double ***partial_field, double ***partial_field_clipboard) {
	double delta_a;
	double **swap;
	double maxdiff = 0;
	int i,j;
	#pragma omp parallel for private(delta_a, j, i) reduction(max: maxdiff) schedule(dynamic, 1) default(shared)
	for(
		i = (neighbor_dim0_left == MPI_PROC_NULL) ? 2 : 1; // catch edges
		i < pi.end_m - pi.start_m + ((neighbor_dim0_right == MPI_PROC_NULL) ? 1 : 2); 
		i++
	) {
		for(
			j = (neighbor_dim1_left == MPI_PROC_NULL) ? 2 : 1; // catch edges
			j < pi.end_n - pi.start_n + ((neighbor_dim1_right == MPI_PROC_NULL) ? 1 : 2); 
			j++
		) {
			delta_a = alpha * 
				    ( ((*partial_field)[i+1][j] + (*partial_field)[i-1][j] - 2.0 * (*partial_field)[i][j]) / (hy_square)
					 +((*partial_field)[i][j-1] + (*partial_field)[i][j+1] - 2.0 * (*partial_field)[i][j]) / (hx_square) );
			delta_a = delta_a * delta_t;
			(*partial_field_clipboard)[i][j] = (*partial_field)[i][j] + delta_a;
			
			if(delta_a > maxdiff)
				maxdiff = delta_a;
		}
	}
	// just switch pointer instead of data
	swap = *partial_field_clipboard;
	*partial_field_clipboard = *partial_field;
	*partial_field = swap;

	return maxdiff <= eps;
}